"""
Module to get PyRunner simulated data from a csv file

HOW TO USE
----------
Create a csv file 'sim_data.csv' in the same folder as this script.
In the HAL.sim configuration file set the setting [Application].MeasSimPath
to the path of this file.

Each line of sim_data.csv must be of the following format:
<well column>,<well row>,<channel 1 signal>[,<channel 2 signal>]
e.g.,
1,1,1200
1,2,1300
"""


import csv
import os
from meas_services.trf import TRFSignalData
from meas_services.fp import FPData
from meas_services.fi import FIData
from meas_services.abs import ABSFData


def get_value(col, row, x_scan=0, y_scan=0, iteration=1):
    folder = os.path.dirname(os.path.abspath(__file__))
    data_file = os.path.join(folder, 'sim_data.csv')

    with open(data_file, newline='') as csvfile:
        data = csv.reader(csvfile, delimiter=',')
        data = [[int(v) for v in x] for x in data]

        # create rectangle around well A1 and available data and fill missing wells with 0s
        cols = max([x[0] for x in data])
        rows = max([x[1] for x in data])
        well_data = [[0 for i in range(rows)] for j in range(cols)]

        for datapoint in data:
            dp_row = datapoint[0]-1
            dp_col = datapoint[1]-1
            dp_channel1 = datapoint[2]
            try:
                dp_channel2 = datapoint[3]
            except IndexError:
                dp_channel2 = dp_channel1

            well_data[dp_row][dp_col] = (dp_channel1, dp_channel2)

        # if data set is to small for selected well, just repeat data periodically
        col = (col - 1) % cols
        row = (row - 1) % rows

        # some random calculations to get different value for each iteration and wellscan poisition
        channel1, channel2 = well_data[col][row] * iteration

        if x_scan and y_scan:
            channel1 = channel1 * x_scan * y_scan
            channel2 = channel2 * x_scan * y_scan

        return (channel1, channel2)


def trf_meas(xwell, ywell, windows, xscan, yscan, iteration, measurementTime):
    ch1, ch2 = get_value(xwell, ywell, xscan, yscan, iteration)

    pmt1signals = list()
    pmt2signals = list()
    for i in range(windows):
        pmt1signals.append(ch1 * (i+1))
        pmt2signals.append(ch2 * (i+2))
    refSignal = 42

    return TRFSignalData(refSignal, pmt1signals, pmt2signals, ())


def fi_meas(xwell, ywell, xscan, yscan, iteration,  measurementTime):
    ch1, ch2 = get_value(xwell, ywell, xscan, yscan, iteration)
    refSignal = 42
    return FIData(ch1, ch2, refSignal, ())


def fp_meas(xwell, ywell, xscan, yscan, iteration,  measurementTime):
    ch1, ch2 = get_value(xwell, ywell, xscan, yscan, iteration)
    refSignal = 42
    return FPData(ch1, ch2, refSignal, ())


def lum_meas(xwell, ywell, xscan, yscan, iteration,  measurementTime):
    ch1, ch2 = get_value(xwell, ywell, xscan, yscan, iteration)
    return (ch1, ch2)


def uslum_meas(xwell, ywell, xscan, yscan, iteration,  measurementTime):
    ch1, ch2 = get_value(xwell, ywell, xscan, yscan, iteration)
    return (ch1, ch2)


def alpha_meas(xwell, ywell, xscan, yscan, iteration,  excitationTime, detectionTime):
    ch1, ch2 = get_value(xwell, ywell, xscan, yscan, iteration)
    return (ch1, ch2)


def alphaus_meas(xwell, ywell, xscan, yscan, iteration, excitationTime, detectionTime):
    ch1, ch2 = get_value(xwell, ywell, xscan, yscan, iteration)
    return ch1


def abs_meas(xwell, ywell, xscan, yscan, iteration,  measurementTime):
    background = 4000
    refBackground = 6000
    refSignal = 7000
    signal, _ = get_value(xwell, ywell, xscan, yscan, iteration)

    return ABSFData(signal, refSignal, background, refBackground)

def fi_mono_meas(xwell, ywell, xscan, yscan, iteration,  measurementTime, exitation_wavelength , emmision_wavelength):
    ch1, ch2 = get_value(xwell, ywell, xscan, yscan, iteration)
    refSignal = 42
    return FIData(ch1, ch2, refSignal, ())

def lum_mono_meas(xwell, ywell, xscan, yscan, iteration,  measurementTime, emmision_wavelength):
    ch1, ch2 = get_value(xwell, ywell, xscan, yscan, iteration)
    return (ch1, ch2)

def abs_mon_meas(xwell, ywell, xscan, yscan, iteration,  measurementTime, exitation_wavelength , emmision_wavelength):
    background = 4000
    refBackground = 6000
    refSignal = 7000
    signal, _ = get_value(xwell, ywell, xscan, yscan, iteration)

    return ABSFData(signal, refSignal, background, refBackground)

def sleep_service(delay):
    pass
